/*
 * Decompiled with CFR 0.152.
 */
package dan200.computercraft.core.filesystem;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.io.ByteStreams;
import dan200.computercraft.api.filesystem.FileOperationException;
import dan200.computercraft.api.filesystem.IMount;
import dan200.computercraft.core.apis.handles.ArrayByteChannel;
import dan200.computercraft.core.filesystem.FileSystem;
import dan200.computercraft.shared.util.IoUtil;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileTime;
import java.time.Instant;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import javax.annotation.Nonnull;

public class JarMount
implements IMount {
    private static final int MAX_CACHED_SIZE = 0x100000;
    private static final int MAX_CACHE_SIZE = 0x4000000;
    private static final Cache<FileEntry, byte[]> CONTENTS_CACHE = CacheBuilder.newBuilder().concurrencyLevel(4).expireAfterAccess(60L, TimeUnit.SECONDS).maximumWeight(0x4000000L).weakKeys().weigher((k, v) -> ((byte[])v).length).build();
    private static final ReferenceQueue<JarMount> MOUNT_QUEUE = new ReferenceQueue();
    private final ZipFile zip;
    private final FileEntry root;

    public JarMount(File jarFile, String subPath) throws IOException {
        JarMount.cleanup();
        if (!jarFile.exists() || jarFile.isDirectory()) {
            throw new FileNotFoundException("Cannot find " + jarFile);
        }
        try {
            this.zip = new ZipFile(jarFile);
        }
        catch (IOException e) {
            throw new IOException("Error loading zip file", e);
        }
        if (this.zip.getEntry(subPath) == null) {
            this.zip.close();
            throw new FileNotFoundException("Zip does not contain path");
        }
        new MountReference(this);
        this.root = new FileEntry();
        Enumeration<? extends ZipEntry> zipEntries = this.zip.entries();
        while (zipEntries.hasMoreElements()) {
            ZipEntry entry = zipEntries.nextElement();
            String entryPath = entry.getName();
            if (!entryPath.startsWith(subPath)) continue;
            String localPath = FileSystem.toLocal(entryPath, subPath);
            this.create(entry, localPath);
        }
    }

    private FileEntry get(String path) {
        FileEntry lastEntry = this.root;
        int lastIndex = 0;
        while (lastEntry != null && lastIndex < path.length()) {
            int nextIndex = path.indexOf(47, lastIndex);
            if (nextIndex < 0) {
                nextIndex = path.length();
            }
            lastEntry = lastEntry.children == null ? null : lastEntry.children.get(path.substring(lastIndex, nextIndex));
            lastIndex = nextIndex + 1;
        }
        return lastEntry;
    }

    private void create(ZipEntry entry, String localPath) {
        FileEntry lastEntry = this.root;
        int lastIndex = 0;
        while (lastIndex < localPath.length()) {
            FileEntry nextEntry;
            int nextIndex = localPath.indexOf(47, lastIndex);
            if (nextIndex < 0) {
                nextIndex = localPath.length();
            }
            String part = localPath.substring(lastIndex, nextIndex);
            if (lastEntry.children == null) {
                lastEntry.children = new HashMap<String, FileEntry>(0);
            }
            if ((nextEntry = lastEntry.children.get(part)) == null || !nextEntry.isDirectory()) {
                nextEntry = new FileEntry();
                lastEntry.children.put(part, nextEntry);
            }
            lastEntry = nextEntry;
            lastIndex = nextIndex + 1;
        }
        lastEntry.setup(entry);
    }

    @Override
    public boolean exists(@Nonnull String path) {
        return this.get(path) != null;
    }

    @Override
    public boolean isDirectory(@Nonnull String path) {
        FileEntry file = this.get(path);
        return file != null && file.isDirectory();
    }

    @Override
    public void list(@Nonnull String path, @Nonnull List<String> contents) throws IOException {
        FileEntry file = this.get(path);
        if (file == null || !file.isDirectory()) {
            throw new FileOperationException(path, "Not a directory");
        }
        file.list(contents);
    }

    @Override
    public long getSize(@Nonnull String path) throws IOException {
        FileEntry file = this.get(path);
        if (file != null) {
            return file.size;
        }
        throw new FileOperationException(path, "No such file");
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    @Nonnull
    public ReadableByteChannel openForRead(@Nonnull String path) throws IOException {
        FileEntry file = this.get(path);
        if (file == null) throw new FileOperationException(path, "No such file");
        if (file.isDirectory()) throw new FileOperationException(path, "No such file");
        byte[] contents = (byte[])CONTENTS_CACHE.getIfPresent((Object)file);
        if (contents != null) {
            return new ArrayByteChannel(contents);
        }
        try {
            ZipEntry entry = this.zip.getEntry(file.path);
            if (entry == null) throw new FileOperationException(path, "No such file");
            try (InputStream stream = this.zip.getInputStream(entry);){
                if (stream.available() > 0x100000) {
                    ReadableByteChannel readableByteChannel = Channels.newChannel(stream);
                    return readableByteChannel;
                }
                contents = ByteStreams.toByteArray((InputStream)stream);
                CONTENTS_CACHE.put((Object)file, (Object)contents);
                ArrayByteChannel arrayByteChannel = new ArrayByteChannel(contents);
                return arrayByteChannel;
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
        throw new FileOperationException(path, "No such file");
    }

    @Override
    @Nonnull
    public BasicFileAttributes getAttributes(@Nonnull String path) throws IOException {
        ZipEntry entry;
        FileEntry file = this.get(path);
        if (file != null && (entry = this.zip.getEntry(file.path)) != null) {
            return new ZipEntryAttributes(entry);
        }
        throw new FileOperationException(path, "No such file");
    }

    private static void cleanup() {
        Reference<JarMount> next;
        while ((next = MOUNT_QUEUE.poll()) != null) {
            IoUtil.closeQuietly(((MountReference)next).file);
        }
    }

    private static class ZipEntryAttributes
    implements BasicFileAttributes {
        private final ZipEntry entry;
        private static final FileTime EPOCH = FileTime.from(Instant.EPOCH);

        ZipEntryAttributes(ZipEntry entry) {
            this.entry = entry;
        }

        @Override
        public FileTime lastModifiedTime() {
            return ZipEntryAttributes.orEpoch(this.entry.getLastModifiedTime());
        }

        @Override
        public FileTime lastAccessTime() {
            return ZipEntryAttributes.orEpoch(this.entry.getLastAccessTime());
        }

        @Override
        public FileTime creationTime() {
            FileTime time = this.entry.getCreationTime();
            return time == null ? this.lastModifiedTime() : time;
        }

        @Override
        public boolean isRegularFile() {
            return !this.entry.isDirectory();
        }

        @Override
        public boolean isDirectory() {
            return this.entry.isDirectory();
        }

        @Override
        public boolean isSymbolicLink() {
            return false;
        }

        @Override
        public boolean isOther() {
            return false;
        }

        @Override
        public long size() {
            return this.entry.getSize();
        }

        @Override
        public Object fileKey() {
            return null;
        }

        private static FileTime orEpoch(FileTime time) {
            return time == null ? EPOCH : time;
        }
    }

    private static class MountReference
    extends WeakReference<JarMount> {
        final ZipFile file;

        MountReference(JarMount file) {
            super(file, MOUNT_QUEUE);
            this.file = file.zip;
        }
    }

    private static class FileEntry {
        String path;
        long size;
        Map<String, FileEntry> children;

        private FileEntry() {
        }

        void setup(ZipEntry entry) {
            this.path = entry.getName();
            this.size = entry.getSize();
            if (this.children == null && entry.isDirectory()) {
                this.children = new HashMap<String, FileEntry>(0);
            }
        }

        boolean isDirectory() {
            return this.children != null;
        }

        void list(List<String> contents) {
            if (this.children != null) {
                contents.addAll(this.children.keySet());
            }
        }
    }
}

